program SOLVEDES;
{--------------------------------------------------------------------}
{  Alg9'1-4.pas   Pascal program for implementing Algorithm 9.1-4    }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 9.1 (Euler's Method).                                   }
{  Section 9.2,   Euler's Method, Page 435                           }
{                                                                    }
{  Algorithm 9.2 (Heun's Method).                                    }
{  Section 9.3,   Heun's Method, Page 441                            }
{                                                                    }
{  Algorithm 9.3 (Taylor's Method of Order 4).                       }
{  Section 9.4,   Taylor Series Method, Page 448                     }
{                                                                    }
{  Algorithm 9.4 (Runge-Kutta Method of Order 4).                    }
{  Section 9.5,   Runge-Kutta Methods, Page 460                      }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    GNmax = 630;
    MaxM = 630;
    FunMax = 9;

  type
    VECTOR = array[0..MaxM] of real;
    DVECTOR = array[1..4] of real;
    LETTER = string[8];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);
    LETTERS = string[200];

  var
    FunType, GNpts, Inum, M, Mend, Meth, Order, Sub: integer;
    A, B, Rnum, Y0: real;
    Ans: CHAR;
    T, X, Y: VECTOR;
    D: DVECTOR;
    State: States;
    DoMo: DoSome;
    Mess: LETTERS;

  function F (T, Y: real): real;
  begin
    case FunType of
      1:
        F := (T - Y) / 2;
      2:
        F := (Y - T) / 2;
      3:
        F := T * T - Y;
      4:
        F := 3 * T + 3 * Y;
      5: 
        F := -T * Y;
      6: 
        F := EXP(-2 * T) - 2 * Y;
      7:
        F := 2 * T * Y * Y;
      8: 
        F := 1 + Y * Y;
      9: 
        F := T * T + Y * Y;
    end;
  end;

  procedure DERIVATIVES (T, Y: real; var D: DVECTOR);
  begin
    case FunType of
      1:
        begin
          D[1] := (T - Y) / 2;
          D[2] := (2 - T + Y) / 4;
          D[3] := (-2 + T - Y) / 8;
          D[4] := (2 - T + Y) / 16;
        end;
      2: 
        begin
          D[1] := (Y - T) / 2;
          D[2] := (Y - T - 2) / 4;
          D[3] := (Y - T - 2) / 8;
          D[4] := (Y - T - 2) / 16;
        end;
      3: 
        begin
          D[1] := T * T - Y;
          D[2] := -T * T + 2 * T + Y;
          D[3] := T * T - 2 * T + 2 - Y;
          D[4] := -T * T + 2 * T - 2 + Y;
        end;
      4: 
        begin
          D[1] := 3 * T + 3 * Y;
          D[2] := 9 * T + 3 + 9 * Y;
          D[3] := 27 * T + 9 + 27 * Y;
          D[4] := 81 * T + 27 + 81 * Y;
        end;
      5:
        begin
          D[1] := -T * Y;
          D[2] := T * T * Y - Y;
          D[3] := -T * T * T * Y + 3 * T * Y;
          D[4] := T * T * T * T * Y - 6 * T * T * Y + 3 * Y;
        end;
      6: 
        begin
          D[1] := EXP(-2 * T) - 2 * Y;
          D[2] := -4 * EXP(-2 * T) + 4 * Y;
          D[3] := 12 * EXP(-2 * T) - 8 * Y;
          D[4] := -32 * EXP(-2 * T) + 16 * Y;
        end;
      7: 
        begin
          D[1] := 2 * T * Y * Y;
          D[2] := 8 * T * T * Y * Y * Y + 2 * Y * Y;
          D[3] := 48 * T * T * T * Y * Y * Y * Y + 24 * T * Y * Y * Y;
          D[4] := 384 * T * T * T * T * Y * Y * Y * Y * Y + 288 * T * T * Y * Y * Y * Y + 24 * Y * Y * Y;
        end;
      8: 
        begin
          D[1] := 1 + Y * Y;
          D[2] := 2 * Y + 2 * Y * Y * Y;
          D[3] := 2 + 8 * Y * Y + 6 * Y * Y * Y * Y;
          D[4] := 16 * Y + 40 * Y * Y * Y + 24 * Y * Y * Y * Y * Y;
        end;
      9:
        begin
          D[1] := T * T + Y * Y;
          D[2] := 2 * T + 2 * T * T * Y + 2 * Y * Y * Y;
          D[3] := 2 + 2 * T * T * T * T + 4 * T * Y + 8 * T * T * Y * Y + 6 * Y * Y * Y * Y;
          D[4] := 8 * T * T * T + 4 * Y + 16 * T * Y * Y + (4 * T + 16 * T * T * Y + 24 * Y * Y * Y) * (T * T + Y * Y);
        end;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1: 
        WRITELN('Y` = (T-Y)/2');
      2: 
        WRITELN('Y` = (Y-T)/2');
      3:
        WRITELN('Y` = T^2 - Y');
      4: 
        WRITELN('Y` = 3*T + 3*Y');
      5: 
        WRITELN('Y` = -T*Y');
      6: 
        WRITELN('Y` = EXP(-2*T) - 2*Y');
      7: 
        WRITELN('Y` = 2*T*Y^2');
      8: 
        WRITELN('Y` = 1 + Y^2');
      9:
        WRITELN('Y` = T^2 + Y^2');
    end;
  end;

  procedure EULER ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E13;
    var
      K: integer;
      H: real;
  begin
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    for K := 0 to M - 1 do
      begin
        Y[K + 1] := Y[K] + H * F(T[K], Y[K]);
        T[K + 1] := A + H * (K + 1);
        if Big < ABS(Y[K + 1]) then
          goto 999;
      end;
999:
    Mend := K + 1;
  end;

  procedure HEUN ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E13;
    var
      J: integer;
      H, K1, K2, P: real;
  begin
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    for J := 0 to M - 1 do
      begin
        K1 := F(T[J], Y[J]);
        P := Y[J] + H * K1;
        T[J + 1] := A + H * (J + 1);
        if Big < ABS(P) then
          begin
            Y[J + 1] := P;
            goto 999;
          end;
        K2 := F(T[J + 1], P);
        Y[J + 1] := Y[J] + H * (K1 + K2) / 2;
        if Big < ABS(Y[J + 1]) then
          goto 999;
      end;
999:
    Mend := J + 1;
  end;

  procedure CAUCHY ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E13;
    var
      J: integer;
      H, K1, K2, P, TP: real;
  begin
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    for J := 0 to M - 1 do
      begin
        K1 := F(T[J], Y[J]);
        P := Y[J] + H * K1 / 2;
        T[J + 1] := A + H * (J + 1);
        if Big < ABS(P) then
          begin
            Y[J + 1] := P;
            goto 999;
          end;
        TP := T[J] + H / 2;
        K2 := F(TP, P);
        Y[J + 1] := Y[J] + H * K2;
        if Big < ABS(Y[J + 1]) then
          goto 999;
      end;
999:
    Mend := J + 1;
  end;

  procedure TAYLOR ({PROCEDURE DERIVATIVES(T,Y: real;}
                                    {VAR D: DVECTOR);}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E13;
    var
      K: integer;
      H, TK, YK: real;
  begin
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    for K := 0 to M - 1 do
      begin
        TK := T[K];
        YK := Y[K];
        DERIVATIVES(TK, YK, D);
        if Order = 3 then
          Y[K + 1] := YK + H * (D[1] + H * (D[2] / 2 + H * D[3] / 6));
        if Order = 4 then
          Y[K + 1] := YK + H * (D[1] + H * (D[2] / 2 + H * (D[3] / 6 + H * D[4] / 24)));
        T[K + 1] := A + H * (K + 1);
        if Big < ABS(Y[K + 1]) then
          goto 999;
      end;
999:
    Mend := K + 1;
  end;

  procedure RUNGE ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E13;
    var
      J: integer;
      H, K1, K2, K3, K4, TJ, YJ: real;
    procedure POLE;                              {Euler or Heun's method}
      label
        990;
      var
        P: real;
    begin
      P := YJ + K1;
      T[J + 1] := A + H * (J + 1);
      if Big < ABS(P) then
        begin
          Y[J + 1] := P;
          goto 990;
        end;
      K2 := F(T[J + 1], P);
      Y[J + 1] := YJ + H * (K1 + K2) / 2;
990:
    end;                                          {End of Procedure Pole}
  begin
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    for J := 0 to M - 1 do
      begin
        TJ := T[J];
        YJ := Y[J];
        K1 := H * F(TJ, YJ);
        if Big < ABS(YJ + 0.5 * K1) then
          begin
            POLE;
            goto 999;
          end;
        K2 := H * F(TJ + H / 2, YJ + 0.5 * K1);
        if Big < ABS(YJ + 0.5 * K2) then
          begin
            POLE;
            goto 999;
          end;
        K3 := H * F(TJ + H / 2, YJ + 0.5 * K2);
        if Big < ABS(YJ + K3) then
          begin
            POLE;
            goto 999;
          end;
        K4 := H * F(TJ + H, YJ + K3);
        Y[J + 1] := YJ + (K1 + 2 * K2 + 2 * K3 + K4) / 6;
        T[J + 1] := A + H * (J + 1);
        if Big < ABS(Y[J + 1]) then
          goto 999;
      end;
999:
    Mend := J + 1;
  end;

  procedure MESSAGE (var Meth: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('                      SOLUTION OF DIFFERENTIAL EQUATIONS');
    WRITELN;
    WRITELN;
    WRITELN('          Solution of the differential equation y` = f(t,y) with the ');
    WRITELN;
    WRITELN('          initial condition  y(a) = y   will be computed over  [a,b].');
    WRITELN('                                     0  ');
    WRITELN;
    WRITELN;
    WRITELN('                      Choose the method of approximation:');
    WRITELN;
    WRITELN('                      < 1 >  Euler`s method');
    WRITELN;
    WRITELN('                      < 2 >  Modified Euler-Cauchy method');
    WRITELN;
    WRITELN('                      < 3 >  Heun`s method');
    WRITELN;
    WRITELN('                      < 4 >  Taylor`s method of order N=3');
    WRITELN;
    WRITELN('                      < 5 >  Taylor`s method of order N=4');
    WRITELN;
    WRITELN('                      < 6 >  Runge-Kutta method of order N=4');
    WRITELN;
    Mess := '                      SELECT < 1 - 6 > ?  ';
    Meth := 1;
    WRITE(Mess);
    READLN(Meth);
    if (Meth < 1) and (State <> Changes) then
      Meth := 1;
    if (Meth > 6) and (State <> Changes) then
      Meth := 6;
    if Meth = 4 then
      Order := 3;
    if Meth = 5 then
      Order := 4;
  end;

  procedure INPUT (var FunType: integer; var A, B, Y0: real; var M: integer; MaxM: integer);
    var
      K: integer;
      Ans: LETTER;
  begin
    CLRSCR;
    WRITELN;
    WRITE('       You chose ');
    case Meth of
      1: 
        WRITE('Euler`s method');
      2: 
        WRITE('the Modified Euler-Cauchy method');
      3: 
        WRITE('Heun`s method');
      4: 
        WRITE('Taylor`s method of order N=3');
      5: 
        WRITE('Taylor`s method of order N=4');
      6: 
        WRITE('the Runge-Kutta method of order N=4');
    end;
    WRITELN(' to solve Y` = F(T,Y).');
    WRITELN;
    WRITELN('                          Choose your D.E.:');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('                     <', K : 1, '>  ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '                          SELECT < 1 - 9 > ?  ';
    FunType := 1;
    WRITE(Mess);
    READLN(FunType);
    if (FunType < 1) and (State <> Changes) then
      FunType := 1;
    if (FunType > FunMax) and (State <> Changes) then
      FunType := FunMax;
    CLRSCR;
    WRITELN;
    WRITE('          You chose ');
    case Meth of
      1: 
        WRITE('Euler`s method');
      2: 
        WRITE('the Modified Euler-Cauchy method');
      3:
        WRITE('Heun`s method');
      4: 
        WRITE('Taylor`s method of order N=3');
      5: 
        WRITE('Taylor`s method of order N=4');
      6: 
        WRITE('the Runge-Kutta method of order N=4');
    end;
    WRITELN(' to solve the D.E.');
    WRITELN;
    WRITE('                    ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('          with the initial condition  Y(A) = Y0.  A numerical');
    WRITELN;
    WRITELN('          approximation is computed over [A,B].  You must give');
    WRITELN;
    WRITELN('          the endpoints for the interval, the initial condition, ');
    WRITELN;
    WRITELN('          and the number of steps.');
    WRITELN;
    WRITELN;
    WRITE('          Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure EPOINTS (var A, B, Y0: real; var M: integer; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      I: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        CLRSCR;
        WRITELN;
        WRITE('                 ');
        PRINTFUNCTION(FunType);
        WRITELN;
        WRITELN;
        if (Stat = Enter) then
          begin
            Mess := '     ENTER  the  left  endpoint  A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            Mess := '     ENTER  the  right endpoint  B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
            Mess := '     ENTER initial condition  Y(A) = ';
            WRITE(Mess);
            READLN(Y0);
            WRITELN;
            Mess := '     ENTER the number of steps   M = ';
            M := 1;
            WRITE(Mess);
            READLN(M);
            WRITELN;
            if M < 1 then
              M := 1;
            if M > 1000 then
              M := 1000;
          end
        else
          begin
            WRITELN('     The  left  endpoint  is     A =', A : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The  right endpoint  is     B =', B : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     Initial   condition  is  Y(A) =', Y0 : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The number of steps  is     M =  ', M : 2);
          end;
        WRITELN;
        WRITELN;
        WRITE('     Want to make a change ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            CLRSCR;
            WRITELN;
            WRITE('                 ');
            PRINTFUNCTION(FunType);
            WRITELN;
            WRITELN('     The current left  endpoint is A =', A : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('     The current right endpoint is B =', B : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
            WRITELN('     The   current   I. C.  is  Y(A) =', Y0 : 15 : 7);
            Mess := '     Now  ENTER the NEW  I. C.  Y(A) = ';
            WRITE(Mess);
            READLN(Y0);
            WRITELN;
            WRITELN('     The  current value of  M  is  M =  ', M : 2);
            Mess := '     Now  ENTER  the NEW value of  M = ';
            WRITE(Mess);
            READLN(M);
            if (M < 1) then
              M := 1;
            if (M > 1000) then
              M := 1000;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RESULTS (FunType: integer; T, Y: VECTOR; M, Mend: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    case Meth of
      1:
        WRITE('Euler`s method');
      2:
        WRITE('The Modified Euler-Cauchy method');
      3:
        WRITE('Heun`s method');
      4:
        WRITE('Taylor`s method of order N=3');
      5:
        WRITE('Taylor`s method of order N=4');
      6:
        WRITE('The Runge-Kutta method of order N=4');
    end;
    WRITELN(' was used to solve the D.E.');
    WRITELN;
    WRITE('      ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('with  Y(', T[0] : 15 : 7, '  ) =', Y[0] : 15 : 7);
    WRITELN;
    WRITELN('    K', '       T(K)          ', '         Y(K)');
    WRITELN('  ------------------------------------------------');
    WRITELN;
    for K := 0 to Mend do
      begin
        WRITELN(K : 5, '   ', T[K] : 15 : 7, '     ', Y[K] : 15 : 7);
        WRITELN;
        if K mod 11 = 9 then
          begin
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
            WRITELN;
          end;
      end;
    if Mend < M then
      begin
        WRITELN('The solution points are approaching a pole.');
        WRITELN;
      end;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  A := 0;
  B := 1;
  Y0 := 0;
  M := 1;
  State := Working;
  while Meth <> 0 do
    begin
      MESSAGE(Meth);
      DoMo := Go;
      while DoMo = Go do
        begin
          INPUT(FunType, A, B, Y0, M, MaxM);
          while (State = Working) or (State = Changes) do
            begin
              EPOINTS(A, B, Y0, M, State);
              case Meth of
                1: 
                  EULER(A, B, Y0, M, Mend, T, Y);
                2: 
                  CAUCHY(A, B, Y0, M, Mend, T, Y);
                3: 
                  HEUN(A, B, Y0, M, Mend, T, Y);
                4: 
                  TAYLOR(A, B, Y0, M, Mend, T, Y);
                5: 
                  TAYLOR(A, B, Y0, M, Mend, T, Y);
                6: 
                  RUNGE(A, B, Y0, M, Mend, T, Y);
              end;
              RESULTS(FunType, T, Y, M, Mend);
              WRITELN;
              WRITELN;
              WRITE('Want to use a  different  initial condition ?  <Y/N>  ');
              READLN(Ans);
              WRITELN;
              if (Ans <> 'Y') and (Ans <> 'y') then
                State := Done
              else
                State := Changes;
            end;
          WRITELN;
          WRITE('Want to  change  the  differential equation ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            DoMo := Stop
          else
            State := Changes;
        end;
      WRITELN;
      WRITE('Want to try another method of approximation ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Meth := 0
      else
        State := Changes;
    end;
end.                                            {End of Main Program}

